
/**
 * Class: BatchValidation.java
 */
package com.cra.pao.fvu;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;

import com.cra.contr.common.util.CommonValidation;
import com.cra.contr.common.util.encrypt.LuhnsAlgorithm;
import com.cra.pao.vo.PAOContrErrorFileVO;

import cra.standalone.paosubcontr.PAOFvu;

/**
 * This class contains methods that will validate the fields of Batch Header Record. It implements FvuConstants
 * interface for getting error descriptions for the corresponding error codes and for other constant values.
 * 
 * @author TCS
 */


public class BatchValidation
{
    private String AIN = null; // Accounts Office Identification Number
    private String fileCreationDate = null; // Date of File Creation
    private String batchID = null; // Batch ID (Unique for a file)
    private String transactionId = null; // Unique Transaction Id of a file
    private String tierTwoTotalSelfContributionOfDDO = null; // Self Contribution Amount Tier-2
    private boolean invalidBH_AIN = false;
    private String bhRecordHash = null;
    private boolean totGovtContrFlag=false;
    private boolean zeroselfContrFlag=false;
    private boolean selfContrFlag=false;
    
    /**
     * ***************************************** BATCH HEADER VALIDATION STARTS***************************************
     */
    /**
     * bhFieldValidator method is called from SubContrFileFormatValidator class to validate the fields present in the
     * Batch Header Record.
     * 
     * @param objReadBH (Reference for SubContrFileFormatValidator class)
     * @param lineCountP (Logical line number for each line in the file)
     * @param bhRecord (The Batch Header Record is taken as a String, line no.2 of the file)
     * @param listErrorVO (Reference for ArrayList VO)
     * 
     * @return void
     */
    void bhFieldValidator(SubContrFileFormatValidator objReadBH, int lineCountP, String[] bhRecord, ArrayList<PAOContrErrorFileVO> listErrorVO ,int utilityLevel)
    {
        if (objReadBH.inValidBhCaretCount == false)
        {
            objReadBH.lineNo = bhRecord[0];
            objReadBH.recType = bhRecord[1];
            objReadBH.batchNo = bhRecord[2];
            objReadBH.typeOfUploadedFile = bhRecord[3];
            AIN = bhRecord[4];
            fileCreationDate = bhRecord[5];
            batchID = bhRecord[6];
            transactionId = bhRecord[7];
            objReadBH.totalCountOfDDO = bhRecord[8];
            objReadBH.totalCountOfSubscribers = bhRecord[9];
            objReadBH.batchTotalGovtContrbutionOfDDO = bhRecord[10];
            objReadBH.batchTotalSelfContrbutionOfDDO = bhRecord[11];
            tierTwoTotalSelfContributionOfDDO = bhRecord[12];
            objReadBH.totalOfTierOneTierTwo = bhRecord[13];
            bhRecordHash = bhRecord[14];
         
//          Decide How the field name will be displayed on the basis of Upload By P / U
            if(objReadBH.uploadBy.equals(FvuConstants.UPLOADED_BY_UOS))
            {
            	objReadBH.PAO_BH_FIELD_5 = FvuConstants.PAO_BH_FIELD[16];
            	objReadBH.PAO_BH_FIELD_9 = FvuConstants.PAO_BH_FIELD[17];
            	objReadBH.PAO_BH_FIELD_11 = FvuConstants.PAO_BH_FIELD[18];
            	objReadBH.PAO_BH_FIELD_12 = FvuConstants.PAO_BH_FIELD[19];
            	
            }//Added To TTS CR-34025.START
            else if(objReadBH.uploadBy.equals(FvuConstants.UPLOADED_BY_UOS_TTS))
            {
            	objReadBH.PAO_BH_FIELD_5 = FvuConstants.PAO_BH_FIELD[16];
            	objReadBH.PAO_BH_FIELD_9 = FvuConstants.PAO_BH_FIELD[17];
            	objReadBH.PAO_BH_FIELD_11 = FvuConstants.PAO_BH_FIELD[18];
            	objReadBH.PAO_BH_FIELD_12 = FvuConstants.PAO_BH_FIELD[19];
            	
            }//Added To TTS CR-34025.END
            //Modified for CR-58405
            else if(objReadBH.uploadBy.equals(FvuConstants.UPLOADED_BY_AO) || objReadBH.uploadBy.equals(FvuConstants.UPLOADED_BY_AO_UPS))
            {
            	objReadBH.PAO_BH_FIELD_5 = FvuConstants.PAO_BH_FIELD[5];
            	objReadBH.PAO_BH_FIELD_9 = FvuConstants.PAO_BH_FIELD[9];
            	objReadBH.PAO_BH_FIELD_11 = FvuConstants.PAO_BH_FIELD[11];
            	objReadBH.PAO_BH_FIELD_12 = FvuConstants.PAO_BH_FIELD[12];
            	totGovtContrFlag=true;
            	zeroselfContrFlag=true;
            	selfContrFlag=true;
            }
            else if (objReadBH.uploadBy.equals(FvuConstants.UPLOADED_BY_KA))
            {
            	objReadBH.PAO_BH_FIELD_5 = FvuConstants.PAO_BH_FIELD[5];
            	objReadBH.PAO_BH_FIELD_9 = FvuConstants.PAO_BH_FIELD[9];
            	objReadBH.PAO_BH_FIELD_11 = FvuConstants.PAO_BH_FIELD[11];
            	objReadBH.PAO_BH_FIELD_12 = FvuConstants.PAO_BH_FIELD[12];
            	
            }
            //Modified for CR-58405
            else if(objReadBH.uploadBy.equals(FvuConstants.UPLOADED_BY_PRAO) || objReadBH.uploadBy.equals(FvuConstants.UPLOADED_BY_PRAO_UPS))
            {
            	objReadBH.PAO_BH_FIELD_5 = FvuConstants.PAO_BH_FIELD[20];
            	objReadBH.PAO_BH_FIELD_9 = FvuConstants.PAO_BH_FIELD[21];
            	objReadBH.PAO_BH_FIELD_11 = FvuConstants.PAO_BH_FIELD[11];
            	objReadBH.PAO_BH_FIELD_12 = FvuConstants.PAO_BH_FIELD[12];
            	totGovtContrFlag=true;
            	zeroselfContrFlag=true;
            	selfContrFlag=true;
            }
            else if(objReadBH.uploadBy.equals(FvuConstants.UPLOADED_BY_CORP))
            {
            	objReadBH.PAO_BH_FIELD_5 = FvuConstants.PAO_BH_FIELD[22];
            	objReadBH.PAO_BH_FIELD_9 = FvuConstants.PAO_BH_FIELD[23];
            	objReadBH.PAO_BH_FIELD_11 = FvuConstants.PAO_BH_FIELD[18];
            	objReadBH.PAO_BH_FIELD_12 = FvuConstants.PAO_BH_FIELD[19];
            	
            }
            /**
             * Validation of LINE NUMBER(Field No.1 of Batch Header Record)
             * 
             * Line Number should not be NULL. Line Number should be of length less than or equal to 9 digits. Line
             * Number should not have leading and trailing spaces. Line Number should not have spaces in between the
             * number. Line Number should always be in sequence.
             */

            if (objReadBH.lineNo == FvuConstants.FIELD_NULL)
            {
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[1], FvuConstants.PEF_2001, FvuConstants.RECTYPE_BHREC, "-", "-","-", "-", "-"));
                objReadBH.lineNo = "-";
            }
            else if (objReadBH.lineNo.trim().length() > 9 || objReadBH.lineNo.length() > 9)
            {
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[1], FvuConstants.PEF_2005, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (objReadBH.lineNo.length() <= 9)
            {
                String bhLineNum = CommonValidation.trimInnerSpaces(objReadBH.lineNo);
                if (CommonValidation.isInt(bhLineNum))
                {
                    listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[1], FvuConstants.PEF_2003, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                }
                else if (!bhLineNum.equals(objReadBH.lineNo))
                {
                    listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[1], FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                }
                else if (Integer.parseInt(objReadBH.lineNo.trim()) != lineCountP)
                {
                    listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[1], FvuConstants.PEF_2004, FvuConstants.RECTYPE_BHREC, "-", "-","-",objReadBH.lineNo, "-"));
                }
            }
            else if (CommonValidation.checkTabSpaces(objReadBH.lineNo))
            {
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[1], FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (objReadBH.lineNo.trim().length() != objReadBH.lineNo.length())
            {
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[1], FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            } // End of LINE NUMBER Validation
            /**
             * Validation of RECORD TYPE(Field No.2 of Batch Header Record)
             * 
             * Record Type should not be NULL. Record Type should be "BH" for Batch Header Record. Values other than
             * "BH" are invalid.
             */
            if (objReadBH.recType == FvuConstants.FIELD_NULL || objReadBH.recType.equals(FvuConstants.FIELD_SEPERATOR))
            {
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[2], FvuConstants.PEF_2001, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (!objReadBH.recType.equals(FvuConstants.BH_REC))
            {
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[2], FvuConstants.PEF_2006, FvuConstants.RECTYPE_BHREC, "-", "-", "-",objReadBH.lineNo, "-"));
            } // End of RECORD TYPE Validation
            /**
             * Validation of BATCH NUMBER(Field No.3 of Batch Header Record)
             * 
             * Batch Number should not be NULL. Batch Number should be of length less than or equal to 9 digits. Batch
             * Number should not have leading and trailing spaces. Batch Number should not have spaces in between the
             * number. Batch Number should always have the value as '1'.
             */
            if (objReadBH.batchNo == FvuConstants.FIELD_NULL)
            {
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[3], FvuConstants.PEF_2001, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (objReadBH.batchNo.trim().length() > 9 || objReadBH.batchNo.length() > 9)
            {
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[3], FvuConstants.PEF_2005, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (objReadBH.batchNo.length() <= 9)
            {
                String batchNumber = CommonValidation.trimInnerSpaces(objReadBH.batchNo);
                if (CommonValidation.isInt(batchNumber))
                {
                    listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[3], FvuConstants.PEF_2003, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                }
                else if (!batchNumber.equals(objReadBH.batchNo))
                {
                    listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[3], FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                }
                else if (Integer.parseInt(objReadBH.batchNo.trim()) != 1)
                {
                    listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[3], FvuConstants.PEF_2007, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                }
            }
            else if (objReadBH.batchNo.trim().length() != objReadBH.batchNo.length())
            {
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[3], FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            } // End of BATCH NUMBER Validation
            /**
             * Validation of TYPE OF CONTRIBUTION FILE(Field No.4 of Batch Header Record)
             * 
             * Contribution File type should be "R" for Regular/Original Contribution File and "C" for Correction
             * Contribution File. Value other than "R" and "C" are invalid.
             */
            boolean invalidContrFileType = false;
            if (objReadBH.typeOfUploadedFile == FvuConstants.FIELD_NULL)
            {
            	invalidContrFileType = true;
            	listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[4], FvuConstants.PEF_2001, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            	objReadBH.controlSheetBuffer.append("-" + FvuConstants.FIELD_SEPERATOR);
            }
            else if (!objReadBH.typeOfUploadedFile.equals(FvuConstants.REG_CONTRIBUTION_FILE) && !objReadBH.typeOfUploadedFile.equals(FvuConstants.CORR_CONTRIBUTION_FILE))
            {
            	invalidContrFileType = true;
            	listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[4], FvuConstants.PEF_2008, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            	objReadBH.controlSheetBuffer.append("-" + FvuConstants.FIELD_SEPERATOR);
            }
            else
            {
                objReadBH.controlSheetBuffer.append(objReadBH.typeOfUploadedFile + FvuConstants.FIELD_SEPERATOR);
            } // End of TYPE OF CONTRIBUTION FILE Validation
            /**
             * Validation of AIN (Field No.5 of Batch Header Record)
             * 
             * 
             */
            /*
             * if (objReadBH.uploadBy.equals(UPLOADED_BY_FC)) {
             */
            if (AIN == FvuConstants.FIELD_NULL)
            {
                listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_5, FvuConstants.PEF_2001, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                objReadBH.controlSheetBuffer.append("0" + FvuConstants.FIELD_SEPERATOR);
                invalidBH_AIN = true;
            }
            else if (AIN.trim().length() != AIN.length())
            {
                listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_5, FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                objReadBH.controlSheetBuffer.append("0" + FvuConstants.FIELD_SEPERATOR);
                invalidBH_AIN = true;
            }
            else if (CommonValidation.checkTabSpaces(AIN))
            {
                listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_5, FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                objReadBH.controlSheetBuffer.append("0" + FvuConstants.FIELD_SEPERATOR);
                invalidBH_AIN = true;
            }
            else if (AIN.trim().length() != 7 || AIN.length() != 7)
            {
                listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_5, FvuConstants.PEF_1008, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                objReadBH.controlSheetBuffer.append("0" + FvuConstants.FIELD_SEPERATOR);
                invalidBH_AIN = true;
            }
            else if (CommonValidation.isInt(AIN))
            {
                listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_5, FvuConstants.PEF_2003, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                objReadBH.controlSheetBuffer.append("0" + FvuConstants.FIELD_SEPERATOR);
                invalidBH_AIN = true;
            }
            else if (CommonValidation.checkID(AIN))
            {
                listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_5, FvuConstants.PEF_2003, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                objReadBH.controlSheetBuffer.append("0" + FvuConstants.FIELD_SEPERATOR);
                invalidBH_AIN = true;
            }
            else if (objReadBH.uploaderId == (FvuConstants.FIELD_NULL) || !objReadBH.invalidAIN == false )
            {
            	 objReadBH.controlSheetBuffer.append("0" + FvuConstants.FIELD_SEPERATOR);
            }
            else if(!AIN.equals(objReadBH.uploaderId))
            {
           		listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_5,Integer.parseInt(objReadBH.sectorCategory + FvuConstants.PEF_2009), FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            	objReadBH.controlSheetBuffer.append("0" + FvuConstants.FIELD_SEPERATOR);
            } 	
            else
            {
            	 objReadBH.controlSheetBuffer.append(AIN + FvuConstants.FIELD_SEPERATOR);
            }
            // End Of PAO ID Validation

            /**
             * Validation of FILE CREATION DATE(Field No.6 of Batch Header Record)
             */
            if (fileCreationDate == FvuConstants.FIELD_NULL)
            {
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[6], FvuConstants.PEF_2001, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (fileCreationDate.trim().length() != 8 || fileCreationDate.length() != 8)
            {
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[6], FvuConstants.PEF_2019, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (CommonValidation.checkTabSpaces(fileCreationDate))
            {
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[6], FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (CommonValidation.isInt(fileCreationDate))
            {
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[6], FvuConstants.PEF_2003, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (CommonValidation.isDate(fileCreationDate))
            {
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[6], FvuConstants.PEF_2003, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else
            {
                String year = fileCreationDate.substring(4, 8);
                if (Integer.parseInt(year) < 2004)
                {
                    listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[6], FvuConstants.PEF_2029, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                }
                else
                {
                    SimpleDateFormat sdf = new SimpleDateFormat("ddMMyyyy");
                    Date dateFromFile = new Date();
                    Date sysDate = new Date();
                    try
                    {
                        dateFromFile = sdf.parse(fileCreationDate);
                        if (dateFromFile.compareTo(sysDate) > 0)
                        {
                            listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[6], FvuConstants.PEF_2028, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                            fileCreationDate = "-";
                        }
                    }
                    catch (java.text.ParseException pe)
                    {
                        System.out.println("Parse Exception while Parsing date: " + pe.toString());
                    }
                    objReadBH.controlSheetBuffer.append(fileCreationDate + FvuConstants.FIELD_SEPERATOR);
                }
            } // End of FILE CREATION DATE Validation

            /**
             * Validation of BATCH ID(Field No.7 of Batch Header Record)
             */
            if (batchID == FvuConstants.FIELD_NULL)
            {
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[7], FvuConstants.PEF_2001, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (batchID.trim().length() != 20 || batchID.length() != 20)
            {
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[7], FvuConstants.PEF_2024, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (CommonValidation.checkTabSpaces(batchID))
            {
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[7], FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if(AIN!= null && (!invalidBH_AIN) && !batchID.substring(0,7).equals(AIN))
            {
            	//need to change error message 
            	listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[7], FvuConstants.PEF_2032, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else
            {
                String firstSubString = batchID.substring(0, 10);
                String secondSubString = batchID.substring(10, 20);
                if (CommonValidation.isInt(firstSubString) || CommonValidation.isInt(secondSubString))
                {
                    listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[7], FvuConstants.PEF_2003, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                }
                else
                {
                    objReadBH.controlSheetBuffer.append(batchID + FvuConstants.FIELD_SEPERATOR);
                }
            }

            /**
             * Validation of TRANSACTION ID(Field No. 8 Of Batch Header Record) This field must be NULL for Regular and
             * MANDATORY For Correction Types. Should be an integer Should have length exactly equal to 10 digits.
             * Should not have TAB spaces.
             */
            
            if (invalidContrFileType == false)
            {
            	if (objReadBH.typeOfUploadedFile.equals(FvuConstants.REG_CONTRIBUTION_FILE))
            	{
            		if (transactionId != FvuConstants.FIELD_NULL)
            		{
            			listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[8], FvuConstants.PEF_2018, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            		}
            		else
            		{
            			objReadBH.controlSheetBuffer.append("-" + FvuConstants.FIELD_SEPERATOR);
            		}
            	}
            	else
            	{
            		if (transactionId == FvuConstants.FIELD_NULL)
            		{
            			listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[8], FvuConstants.PEF_2001, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            		}
            		else if (transactionId.trim().length() != 13 || transactionId.length() != 13)
            		{
            			listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[8], FvuConstants.PEF_2016, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            		}
            		else if (CommonValidation.checkTabSpaces(transactionId))
            		{
            			listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[8], FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            		}
            		else if (CommonValidation.isInt(transactionId))
            		{
            			listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[8], FvuConstants.PEF_2003, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            		}
            		else if (Long.parseLong(transactionId.trim()) == 0)
            		{
            			listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[8], FvuConstants.PEF_2003, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            		}
            		else if (!LuhnsAlgorithm.isValidNumber(transactionId))
            		{
            			listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[8], FvuConstants.PEF_2003, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            		}
            		else
            		{
            			objReadBH.controlSheetBuffer.append(transactionId + FvuConstants.FIELD_SEPERATOR);
            		}
            	}
            }

            /**
             * Validation of COUNT OF DDO RECORDS(Field No. 9 Of Batch Header Record). This field is MANDATORY for
             * Regular & Correction files. Count should be an integer value with length less than or equal to 9 digits.
             * Count should not be "0" (Zero).
             */
           
            if (objReadBH.totalCountOfDDO == FvuConstants.FIELD_NULL)
            {
            	 objReadBH.invalidDDOCount = true;
            	listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_9, FvuConstants.PEF_2001, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (objReadBH.totalCountOfDDO.length() > 9)
            {
            	objReadBH.invalidDDOCount = true;
                listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_9, FvuConstants.PEF_2005, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (objReadBH.totalCountOfDDO.trim().length() != objReadBH.totalCountOfDDO.length())
            {
            	objReadBH.invalidDDOCount = true;
                listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_9, FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (CommonValidation.checkTabSpaces(objReadBH.totalCountOfDDO))
            {
            	objReadBH.invalidDDOCount = true;
                listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_9, FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else
            {
                String bhDdoCount = CommonValidation.trimInnerSpaces(objReadBH.totalCountOfDDO);
                if (CommonValidation.isInt(bhDdoCount))
                {
                	objReadBH.invalidDDOCount = true;
                    listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_9, FvuConstants.PEF_2003, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                }
                else if (!bhDdoCount.equals(objReadBH.totalCountOfDDO))
                {
                	objReadBH.invalidDDOCount = true;
                    listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_9, FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                }
                else if (Integer.parseInt(objReadBH.totalCountOfDDO.trim()) == 0)
                {
                	objReadBH.invalidDDOCount = true;
                	listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_9, Integer.parseInt(objReadBH.sectorCategory +FvuConstants.PEF_2013), FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                }
                else
                {
                    objReadBH.controlSheetBuffer.append(objReadBH.totalCountOfDDO + FvuConstants.FIELD_SEPERATOR);
                } // End of COUNT OF DDO RECORDS Validation
            }

            /**
             * Validation of COUNT OF SUBSCRIBER RECORDS(Field No. 10 Of Batch Header Record) This field is MANDATORY
             * for Regular & Correction files. Count should be an integer value with length less than or equal to 9
             * digits. Count should not be "0" (Zero).
             */
            if (objReadBH.totalCountOfSubscribers == FvuConstants.FIELD_NULL)
            {
                objReadBH.invalidSubCount = true;
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[10], FvuConstants.PEF_2001, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (objReadBH.totalCountOfSubscribers.length() > 9)
            {
                objReadBH.invalidSubCount = true;
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[10], FvuConstants.PEF_2005, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (objReadBH.totalCountOfSubscribers.trim().length() != objReadBH.totalCountOfSubscribers.length())
            {
                objReadBH.invalidSubCount = true;
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[10], FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (CommonValidation.checkTabSpaces(objReadBH.totalCountOfSubscribers))
            {
                objReadBH.invalidSubCount = true;
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[10], FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else
            {
                String bhSubCount = CommonValidation.trimInnerSpaces(objReadBH.totalCountOfSubscribers);
                if (CommonValidation.isInt(bhSubCount))
                {
                    objReadBH.invalidSubCount = true;
                    listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[10], FvuConstants.PEF_2003, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                }
                else if (!bhSubCount.equals(objReadBH.totalCountOfSubscribers))
                {
                    objReadBH.invalidSubCount = true;
                    listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[10], FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                }
                else if (Integer.parseInt(objReadBH.totalCountOfSubscribers.trim()) == 0)
                {
                    objReadBH.invalidSubCount = true;
                    listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[10], Integer.parseInt(objReadBH.sectorCategory +FvuConstants.PEF_2013), FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                }
                else
                {
                    objReadBH.controlSheetBuffer.append(objReadBH.totalCountOfSubscribers + FvuConstants.FIELD_SEPERATOR);
                } // End of COUNT OF SUBSCRIBER RECORDS Validation
            }

            /**
             * Validation of TIER-1 GOVERNMENT CONTRIBUTION AMOUNT(Field No. 11 Of Batch Header Record)  This field is MANDATORY for Regular &
             * Correction Amount must have length less than or equal to 20 digits. Amount should be a positive decimal
             * value. Amount should be greater than 0.00 for Upload By 'P'.Blank spaces should not be specified between the numbers. No leading and trailing spaces should be
             * specified. No tab spaces should be specified.
             */
            if (objReadBH.batchTotalGovtContrbutionOfDDO == FvuConstants.FIELD_NULL)
            {
                objReadBH.invalidTotalContributionInBatch = true;
                listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_11, FvuConstants.PEF_2001, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (objReadBH.batchTotalGovtContrbutionOfDDO.length() > 21 || objReadBH.batchTotalGovtContrbutionOfDDO.trim().length() > 21)
            {
                objReadBH.invalidTotalContributionInBatch = true;
                listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_11, FvuConstants.PEF_2005, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (CommonValidation.checkTabSpaces(objReadBH.batchTotalGovtContrbutionOfDDO))
            {
                objReadBH.invalidTotalContributionInBatch = true;
                listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_11, FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (objReadBH.batchTotalGovtContrbutionOfDDO.trim().length() != objReadBH.batchTotalGovtContrbutionOfDDO.length())
            {
                objReadBH.invalidTotalContributionInBatch = true;
                listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_11, FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (CommonValidation.isDecimalNumber(objReadBH.batchTotalGovtContrbutionOfDDO))
            {
            	objReadBH.invalidTotalContributionInBatch = true;
                listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_11, FvuConstants.PEF_2003, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            /*else if(utilityLevel != 5 && totGovtContrFlag && Double.parseDouble(objReadBH.batchTotalGovtContrbutionOfDDO) == 0.00)
            {
            	objReadBH.invalidTotalContributionInBatch = true;
            	listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_11, FvuConstants.PEF_2031, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }	*/
            else
            {
            	objReadBH.controlSheetBuffer.append(objReadBH.batchTotalGovtContrbutionOfDDO + FvuConstants.FIELD_SEPERATOR);
            	
            }
            
            /**Validation of TIER-1 SELF CONTRIBUTION AMOUNT(Field No. 12 Of Batch Header Record).This field is MANDATORY for Regular &
             * Correction Amount must have length less than or equal to 20 digits. Amount should be a positive decimal
             * value. Amount should be greater than 0.00.
             */
            if (objReadBH.batchTotalSelfContrbutionOfDDO == FvuConstants.FIELD_NULL)
            {
                objReadBH.invalidTotalContributionInBatch = true;
                listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_12, FvuConstants.PEF_2001, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
   		    else if (objReadBH.batchTotalSelfContrbutionOfDDO.length() > 21 || objReadBH.batchTotalSelfContrbutionOfDDO.trim().length() > 21)
			{
			    objReadBH.invalidTotalContributionInBatch = true;
			    listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_12, FvuConstants.PEF_2005, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
			}
			else if (CommonValidation.checkTabSpaces(objReadBH.batchTotalSelfContrbutionOfDDO))
			{
			    objReadBH.invalidTotalContributionInBatch = true;
			    listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_12, FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
			}
			else if (objReadBH.batchTotalSelfContrbutionOfDDO.trim().length() != objReadBH.batchTotalSelfContrbutionOfDDO.length())
			{
			    objReadBH.invalidTotalContributionInBatch = true;
			    listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_12, FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
			}
			else if (CommonValidation.isDecimalNumber(objReadBH.batchTotalSelfContrbutionOfDDO))
			{
			   	objReadBH.invalidTotalContributionInBatch = true;
			    listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_12, FvuConstants.PEF_2003, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
			}
			/*else if(utilityLevel != 5 && zeroselfContrFlag
					&& Double.parseDouble(objReadBH.batchTotalSelfContrbutionOfDDO) == 0.00)
			{
			   	objReadBH.invalidTotalContributionInBatch = true;
			   	listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_12, FvuConstants.PEF_2031, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
			}*/
            //condition added for 14% CR-39795
			else if (utilityLevel != 5  && selfContrFlag
					&&  (!objReadBH.invalidTotalContributionInBatch)  && !objReadBH.batchTotalGovtContrbutionOfDDO.equals(objReadBH.batchTotalSelfContrbutionOfDDO) && PAOFvu.unequalFlag.equalsIgnoreCase("N")&&"N".equalsIgnoreCase(PAOFvu.OSEqualFlag))
            {
            	objReadBH.invalidTotalContributionInBatch = true;
            	listErrorVO.add(new PAOContrErrorFileVO(objReadBH.PAO_BH_FIELD_12,FvuConstants.PEF_2014, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
			else
	        {
	           	objReadBH.controlSheetBuffer.append(objReadBH.batchTotalSelfContrbutionOfDDO + FvuConstants.FIELD_SEPERATOR);
	        }
		            
            /**
             * Validation of TIER-2 SELF CONTRIBUTION AMOUNT(Field No. 13 Of Batch Header Record) This field must be
             * NULL (As Of Now) 28/09/2007
             */
            if (tierTwoTotalSelfContributionOfDDO != FvuConstants.FIELD_NULL)
            {
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[13], FvuConstants.PEF_2027, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }

            /**
             * Validation of TOTAL OF TIER-1 GOVERNMENT CONTRIBUTION AMOUNT TIER-1 SELF CONTRIBUTION AMOUNT and TIER-2
             * SELF CONTRIBUTION AMOUNT (Field No. 14 Of Batch Header Record) This field is MANDATORY for Regular &
             * Correction Amount must have length less than or equal to 20 digits. Amount should be a positive decimal
             * value. Amount should be greater than or equal to 0.00.
             */

            BigDecimal amountOne = null;
            BigDecimal amountTwo = null;
            BigDecimal totalBankAmt =  null;
            if (objReadBH.totalOfTierOneTierTwo == FvuConstants.FIELD_NULL)
            {
                objReadBH.invalidTotalContributionInBatch = true;
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[14], FvuConstants.PEF_2001, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            // 18 digits,followed by a decimal point(.) and two digits after decimal
            else if (objReadBH.totalOfTierOneTierTwo.length() > 21)
            {
                objReadBH.invalidTotalContributionInBatch = true;
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[14], FvuConstants.PEF_2005, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            /**
             * Blank spaces should not be specified between the numbers. No leading and trailing spaces should be
             * specified. No tab spaces should be specified.
             */
            else if (CommonValidation.checkTabSpaces(objReadBH.totalOfTierOneTierTwo))
            {
                objReadBH.invalidTotalContributionInBatch = true;
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[14], FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (CommonValidation.isDecimalNumber(objReadBH.totalOfTierOneTierTwo))
            {
                objReadBH.invalidTotalContributionInBatch = true;
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[14], FvuConstants.PEF_2003, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (objReadBH.totalOfTierOneTierTwo.trim().length() != objReadBH.totalOfTierOneTierTwo.length())
			{
			    objReadBH.invalidTotalContributionInBatch = true;
			    listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[14], FvuConstants.PEF_2002, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
			}
            else if(new BigDecimal(objReadBH.totalOfTierOneTierTwo).compareTo(new BigDecimal(0.00))==0)
            {
            	objReadBH.invalidTotalContributionInBatch = true;
                listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[14], FvuConstants.PEF_2031, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
            }
            else if (objReadBH.invalidTotalContributionInBatch == false)
            {
                BigDecimal totalAmt = new BigDecimal(objReadBH.totalOfTierOneTierTwo);
                amountOne = new BigDecimal(objReadBH.batchTotalGovtContrbutionOfDDO);
                amountTwo = new BigDecimal(objReadBH.batchTotalSelfContrbutionOfDDO);
                totalBankAmt = amountOne.add(amountTwo);
                if (!totalAmt.equals(totalBankAmt))
                {
                    objReadBH.invalidTotalContributionInBatch = true;
                    listErrorVO.add(new PAOContrErrorFileVO(FvuConstants.PAO_BH_FIELD[14], FvuConstants.PEF_2026, FvuConstants.RECTYPE_BHREC, "-", "-","-", objReadBH.lineNo, "-"));
                }
                else
                {
                    objReadBH.controlSheetBuffer.append(objReadBH.totalOfTierOneTierTwo + FvuConstants.FIELD_SEPERATOR);
                }
            }

            /**
             * ************************************BATCH HEADER VALIDATION ENDS**************************************
             */
        }
    }
} // End of bhFieldValidator method
