package com.cra.contr.common.util.file;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

public class CRAFileOperations {

    protected File file;
    protected static final String forwardSlash = "/";
    protected static final String backwardSlash = "\\";
    
    public CRAFileOperations(final String absoluteFileName) {
	file = new File(absoluteFileName);
    }

    /**
     * Tests whether the file or directory denoted by this abstract pathname
     * exists.
     * 
     * @return <code>true</code> if and only if the file or directory denoted
     *         by this abstract pathname exists; <code>false</code> otherwise
     */
    public boolean exists() {
	return file.exists();
    }

    /**
     * Returns the length of the file denoted by this abstract pathname. The
     * return value is unspecified if this pathname denotes a directory.
     * 
     * @return The length, in bytes, of the file denoted by this abstract
     *         pathname, or <code>0L</code> if the file does not exist
     */
    public long getFileSize() {
	return file.length();
    }

    /**
     * Atomically creates a new, empty file named by this abstract pathname if
     * and only if a file with this name does not yet exist. The check for the
     * existence of the file and the creation of the file if it does not exist
     * are a single operation that is atomic with respect to all other
     * filesystem activities that might affect the file.
     * <P>
     * 
     * @return <code>true</code> if the named file does not exist and was
     *         successfully created; <code>false</code> if the named file
     *         already exists
     * 
     * @throws IOException
     *             If an I/O error occurred
     */
    public boolean createNewFile() throws IOException {
	if(!file.getParentFile().exists()){
	    file.getParentFile().mkdirs();
	}
	return file.createNewFile();
    }

    /**
     * Deletes the file or directory denoted by this abstract pathname. If this
     * pathname denotes a directory, then the directory must be empty in order
     * to be deleted.
     * 
     * @return <code>true</code> if and only if the file or directory is
     *         successfully deleted; <code>false</code> otherwise
     */
    public boolean deleteFile() {
	return file.delete();
    }

    /**
     * Copies a source file to a destination file.
     * 
     * @param sourceFile
     * @param destFile
     * @throws IOException
     */
    public static void copyToFile(final String sourceFile, final String destFile)
    throws IOException {
	byte[] buffer = new byte[4096];
	FileInputStream in = new FileInputStream(sourceFile);
	FileOutputStream out = new FileOutputStream(destFile);
	int len;

	while ((len = in.read(buffer)) != -1) {
	    out.write(buffer, 0, len);
	}

	out.close();
	in.close();
    }

    /**
     * This method is usefull only for directories. This method returns the
     * name of the files present in the directory which the instance
     * variable <code>file</code> represents.
     * 
     * @return String array containing the file names in the directory
     *         represented by this.file. Returns null if the this.file is
     *         not a directory
     */
    public String[] getFileNameList(){
	return this.file.list();
    }

    /**
     * 
     * @param absoluteFileName
     * @return This method returns the file name taking absolute file name as input.
     */
    public static String getFileName(String absoluteFileName) {
	if(absoluteFileName == null){
	    return null;
	}
	int index1 = absoluteFileName.lastIndexOf(forwardSlash);
	int index2 = absoluteFileName.lastIndexOf(backwardSlash);
	int lastFileSeparatorIndex = (index1 > index2) ? index1 : index2;
	return absoluteFileName.substring(lastFileSeparatorIndex + 1, absoluteFileName.length());
    }
    
    public static String getDirName(final String absoluteFileName){

	int index1 = absoluteFileName.lastIndexOf(forwardSlash);
	int index2 = absoluteFileName.lastIndexOf(backwardSlash);
	int lastFileSeparatorIndex = (index1 > index2) ? index1 : index2;
	String filePath = absoluteFileName.substring(0,lastFileSeparatorIndex + 1);
	return filePath;
    }
    
    public boolean isDirectory(){
	return file.isDirectory();
    }

    public static String getFileExtension(final String fileName){
	int lastIndex = fileName.lastIndexOf(".");
	return fileName.substring(lastIndex, fileName.length());
    }
    
    public static String getFileNameWithOutExtension(final String fileName){
	String rFileName = getFileName(fileName);
	int lastIndex = rFileName.lastIndexOf(".");
	return rFileName.substring(0, lastIndex);
    }
    
    public void renameTo(String dest){
    	this.file.renameTo(new File(dest));
    }
}