package com.cra.contr.common.util.encrypt;

public class LuhnsAlgorithm {

    /**
     * Checks whether a string of digits is a valid credit card number according
     * to the Luhn algorithm.
     *
     * 1. Starting with the second to last digit and moving left, double the
     *    value of all the alternating digits. For any digits that thus become
     *    10 or more, add their digits together. For example, 1111 becomes 2121,
     *    while 8763 becomes 7733 (from (1+6)7(1+2)3).
     *
     * 2. Add all these digits together. For example, 1111 becomes 2121, then
     *    2+1+2+1 is 6; while 8763 becomes 7733, then 7+7+3+3 is 20.
     *
     * 3. If the total ends in 0 (put another way, if the total modulus 10 is
     *    0), then the number is valid according to the Luhn formula, else it is
     *    not valid. So, 1111 is not valid (as shown above, it comes out to 6),
     *    while 8763 is valid (as shown above, it comes out to 20).
     *
     * @param number the credit card number to validate.
     * @return true if the number is valid, false otherwise.
     */
    public static boolean isValidNumber(String number) {
	int sum = calculateSum(number, false);
	return (sum % 10 == 0);
    }

    public static String getNumberAppendedWithCheckDigit(String number) {
	int sum = calculateSum(number, true);
	
	int sumMod10 = sum % 10;
	if(sumMod10!=0){
	    sumMod10 = 10 - (sum % 10);
	}
	String checkDigit = number + sumMod10;
	return checkDigit;
    }
    
    private static int calculateSum(String numStr, boolean generate){
	
	int sum = 0;

	boolean alternate = generate;
	for (int i = numStr.length() - 1; i >= 0; i--) {
	    int n = Integer.parseInt(numStr.substring(i, i + 1));
	    if (alternate) {
		n *= 2;
		if (n > 9) {
		    n = (n % 10) + 1;
		}
	    }
	    sum += n;
	    alternate = !alternate;
	}
	return sum;
    }
    
    public static void main(String d[]){
	int inputNumber1 = 44666765; // check digit is 1
	int outputNumber1 = Integer.parseInt(getNumberAppendedWithCheckDigit(""+inputNumber1));
	System.out.println("inputNumber1:"+inputNumber1+":result:"+isValidNumber(""+outputNumber1));
	System.out.println("outputNumber1:"+outputNumber1);
	
	long inputNumber2 = 4477470310678009567L;
	long outputNumber2 = Long.parseLong(getNumberAppendedWithCheckDigit(""+inputNumber2));
	System.out.println("inputNumber2:"+inputNumber2+":result:"+isValidNumber(""+outputNumber2));
	System.out.println("outputNumber2:"+outputNumber2);
	
    }
}